#include <Blobs/StdV3D.h>
#include "MatrixTransf.h"


namespace V3D {


//----------------------------------------------------------------------------
const double EPSILON = 1E-7;


template<> const MatrixTransff MatrixTransff::k_mtxIdentity( Vector3f(1,0,0), Vector3f(0,1,0), Vector3f(0,0,1) );
template<> const MatrixTransfd MatrixTransfd::k_mtxIdentity( Vector3f(1,0,0), Vector3f(0,1,0), Vector3f(0,0,1) );

template<> const MatrixTransff MatrixTransff::k_mtxZero( Vector3f(0,0,0), Vector3f(0,0,0), Vector3f(0,0,0) );
template<> const MatrixTransfd MatrixTransfd::k_mtxZero( Vector3f(0,0,0), Vector3f(0,0,0), Vector3f(0,0,0) );




void TridiagonalNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3])
{
	// Householder reduction T = Q^t M Q
	//	 Input:
	//	   mat, symmetric 3x3 matrix M
	//	 Output:
	//	   mat, orthogonal matrix Q
	//	   diag, diagonal entries of T
	//	   subd, subdiagonal entries of T (T is symmetric)


	float fA = avcCols[0].x;  // m[M11]
	float fB = avcCols[1].x;  // m[M12]
	float fC = avcCols[2].x;  // m[M13]
	float fD = avcCols[1].y;  // m[M22]
	float fE = avcCols[2].y;  // m[M23]
	float fF = avcCols[2].z;  // m[M33]

	afDiag[0] = fA;
	afSubDiag[2] = 0.0;
	if ( Abs(fC) >= EPSILON )
	{
		float fLength = Sqrt(fB*fB+fC*fC);
		float fInvLength = 1.0f / fLength;
		fB *= fInvLength;
		fC *= fInvLength;
		float fQ = 2.0f * fB * fE + fC * (fF - fD);
		afDiag[1] = fD+fC*fQ;
		afDiag[2] = fF-fC*fQ;
		afSubDiag[0] = fLength;
		afSubDiag[1] = fE-fB*fQ;

		avcCols[0].x = 1.f;  // m[M11] = 1.0;
		avcCols[1].x = 0.f;  //	m[M12] = 0.0;
		avcCols[2].x = 0.f;  //	m[M13] = 0.0;
		avcCols[0].y = 0.f;  //	m[M21] = 0.0;
		avcCols[1].y = fB;   //	m[M22] = fB;
		avcCols[2].y = fC;   //	m[M23] = fC;
		avcCols[0].z = 0.f;  //	m[M31] = 0.0;
		avcCols[1].z = fC;   //	m[M32] = fC;
		avcCols[2].z = -fB;  //	m[M33] = -fB;
	}
	else
	{
		afDiag[1] = fD;
		afDiag[2] = fF;
		afSubDiag[0] = fB;
		afSubDiag[1] = fE;
		avcCols[0].Set( 1, 0, 0);
		avcCols[1].Set( 0, 1, 0);
		avcCols[2].Set( 0, 0, 1);
	}
}
//----------------------------------------------------------------------------


bool QLAlgorithmNoTemplate(Vector3f avcCols[3], float afDiag[3], float afSubDiag[3])
{
	// QL iteration with implicit shifting to reduce matrix from tridiagonal
	// to diagonal

	for(int32 i0 = 0; i0 < 3; i0++)
	{
		const int32 iMaxIter = 32;
		int32 iIter;
		for( iIter = 0; iIter < iMaxIter; iIter++)
		{
			int32 i1;
			for( i1 = i0; i1 <= 1; i1++)
			{
				float fSum = Abs(afDiag[i1]) +
				            Abs(afDiag[i1+1]);
				if ( Abs(afSubDiag[i1]) + fSum == fSum )
					break;
			}
			if ( i1 == i0 )
				break;

			float fTmp0 = (afDiag[i0 + 1] - afDiag[i0]) / (2.0f * afSubDiag[i0]);

			float fTmp1 = Sqrt(fTmp0 * fTmp0 + 1.0f);
			if ( fTmp0 < 0.0 )
				fTmp0 = afDiag[i1]-afDiag[i0]+afSubDiag[i0]/(fTmp0-fTmp1);
			else
				fTmp0 = afDiag[i1]-afDiag[i0]+afSubDiag[i0]/(fTmp0+fTmp1);

			float fSin = 1.0;
			float fCos = 1.0;
			float fTmp2 = 0.0;
			for( int32 i2 = i1-1; i2 >= i0; i2--)
			{
				float fTmp3 = fSin*afSubDiag[i2];
				float fTmp4 = fCos*afSubDiag[i2];
				if ( fabs(fTmp3) >= fabs(fTmp0) )
				{
					fCos = fTmp0 / fTmp3;
					fTmp1 = Sqrt(fCos * fCos + 1.0f);
					afSubDiag[i2+1] = fTmp3*fTmp1;
					fSin = 1.0f / fTmp1;
					fCos *= fSin;
				}
				else
				{
					fSin = fTmp3 / fTmp0;
					fTmp1 = Sqrt(fSin * fSin + 1.0f);
					afSubDiag[i2+1] = fTmp0*fTmp1;
					fCos = 1.0f / fTmp1;
					fSin *= fCos;
				}
				fTmp0 = afDiag[i2 + 1] - fTmp2;
				fTmp1 = (afDiag[i2] - fTmp0) * fSin + 2.0f * fTmp4 * fCos;
				fTmp2 = fSin * fTmp1;
				afDiag[i2 + 1] = fTmp0 + fTmp2;
				fTmp0 = fCos * fTmp1 - fTmp4;


				const Vector3f vcTmp1 = avcCols[i2];
				const Vector3f vcTmp2 = avcCols[i2+1];
				avcCols[i2  ].x = fCos * vcTmp1.x - fSin*vcTmp2.x;
				avcCols[i2  ].y = fCos * vcTmp1.y - fSin*vcTmp2.y;
				avcCols[i2  ].z = fCos * vcTmp1.z - fSin*vcTmp2.z;

				avcCols[i2+1].x = fSin * vcTmp1.x + fCos*vcTmp2.x;
				avcCols[i2+1].y = fSin * vcTmp1.y + fCos*vcTmp2.y;
				avcCols[i2+1].z = fSin * vcTmp1.z + fCos*vcTmp2.z;

			}
			afDiag[i0] -= fTmp2;
			afSubDiag[i0] = fTmp0;
			afSubDiag[i1] = 0.0;
		}

		if ( iIter == iMaxIter )
		{
			// should not get here under normal circumstances
			return false;
		}
	}

	return true;
}



} // namespaces
